/**********************************************************************
*  WG9914D.c — UART⇔GPIB 转换芯片 WG9914D 轻量级驱动（v1.0.2）
**********************************************************************/
#include "WG9914D.h"
#include <string.h>
#include <stdio.h>
#include <stdarg.h>

/*--------------------------------------------------------------------
 *  可调参数
 *-------------------------------------------------------------------*/
#ifndef WG9914D_LINEBUF_SZ
#define WG9914D_LINEBUF_SZ   64      /* 行、搜索缓冲大小（含 '\0'） */
#endif

/*--------------------------------------------------------------------
 *  静态变量
 *-------------------------------------------------------------------*/
static WG9914D_TxFunc       tx_func   = NULL;
static WG9914D_Delay1msFunc delay_1ms = NULL;

static char idn_str[96] = "WG9914D,WWKJ,SN00000001,V1.0";

static uint8_t rx_buf[WG9914D_LINEBUF_SZ];
static uint16_t rx_idx = 0;

static char     search_buf[WG9914D_LINEBUF_SZ];
static uint16_t search_len = 0;

static const char *exp1 = NULL;
static const char *exp2 = NULL;
static volatile bool waiting = false;
static volatile bool matched = false;

/*--------------------------------------------------------------------
 *  本地工具
 *-------------------------------------------------------------------*/
static void tx_raw(const char *fmt, ...)
{
    if (!tx_func) return;
    char tmp[WG9914D_LINEBUF_SZ];
    va_list ap;
    va_start(ap, fmt);
    int n = vsnprintf(tmp, sizeof(tmp), fmt, ap);
    va_end(ap);
    if (n > 0) tx_func((const uint8_t *)tmp, (uint16_t)n);
}

static bool str_equal(const char *a, const char *b)
{
    return (a && b) ? (strcmp(a, b) == 0) : false;
}

static bool str_contains(const char *hay, const char *needle)
{
    return (hay && needle && *needle) ? (strstr(hay, needle) != NULL) : false;
}

/*--------------------------------------------------------------------
 *  行级解析（*IDN? 及等待匹配）
 *-------------------------------------------------------------------*/
static void handle_line(char *line)
{
    if (str_equal(line, "*IDN?"))
    {
        tx_raw("%s\r\n", idn_str);
        return;
    }

    if (waiting &&
        ((exp1 && str_contains(line, exp1)) ||
         (exp2 && str_contains(line, exp2))))
    {
        matched  = true;
        waiting  = false;
    }
}

/*--------------------------------------------------------------------
 *  外部接口
 *-------------------------------------------------------------------*/
void WG9914D_InputByte(uint8_t ch)
{
    /* 存入行缓冲 --------------------------------------------------*/
    if (rx_idx >= WG9914D_LINEBUF_SZ) rx_idx = 0;
    rx_buf[rx_idx++] = ch;

    /* 等待阶段累积搜索缓冲 --------------------------------------*/
    if (waiting && search_len < WG9914D_LINEBUF_SZ - 1) {
        search_buf[search_len++] = (char)ch;
        search_buf[search_len]   = '\0';

        if ((exp1 && str_contains(search_buf, exp1)) ||
            (exp2 && str_contains(search_buf, exp2))) {
            matched  = true;
            waiting  = false;
        }
    }

    /* 快速 *IDN? 检测（无换行也可触发） ------------------------*/
    if (rx_idx >= 5 &&
        memcmp(&rx_buf[rx_idx - 5], "*IDN?", 5) == 0) {
        tx_raw("%s\r\n", idn_str);
        rx_idx = 0;
        return;
    }

    /* 行结束处理 -------------------------------------------------*/
    if (ch == '\n') {
        rx_buf[rx_idx - 1] = '\0';      /* 去掉 '\n' */
        handle_line((char *)rx_buf);
        rx_idx = 0;
    }
}

void WG9914D_Init(WG9914D_TxFunc       tx_cb,
                  WG9914D_Delay1msFunc delay_cb,
                  const char          *idn_reply)
{
    tx_func   = tx_cb;
    delay_1ms = delay_cb;
    if (idn_reply && *idn_reply)
        strncpy(idn_str, idn_reply, sizeof(idn_str) - 1);

    rx_idx = search_len = 0;
    waiting = matched = false;
}

bool WG9914D_SendLineWait(const char *cmd,
                          const char *expect1,
                          const char *expect2,
                          uint32_t    timeout_ms)
{
    if (!tx_func || !delay_1ms || !cmd)
        return false;

    bool expect_empty = (!expect1 || !*expect1) &&
                        (!expect2 || !*expect2);

    /* ---------- 立即返回模式 ---------- */
    if (expect_empty || timeout_ms == 0) {
        tx_func((const uint8_t *)cmd, (uint16_t)strlen(cmd));
        return true;
    }

    /* ---------- 清空搜索缓冲 ---------- */
    search_len   = 0;
    search_buf[0] = '\0';

    /* ---------- 发送指令 ---------- */
    tx_func((const uint8_t *)cmd, (uint16_t)strlen(cmd));

    /* ---------- 等待匹配 ---------- */
    exp1    = expect1;
    exp2    = expect2;
    matched = false;
    waiting = true;

    for (uint32_t t = 0; t < timeout_ms && waiting; ++t)
        delay_1ms();

    waiting = false;
    return matched;
}
